/*
 *  Copyright (c) 1992, by
 *
 *     Emanuel G. Noik
 *     noik@db.toronto.edu
 *
 *  This program is adapted from code obtained from watcgl.waterloo.edu,
 *  and can be used to generate output for:  GL graphics library, Postscript,
 *  and the Unix Plot program.
 */

/******************************************************************************/

#include	<stdio.h>
#include	<hershey/hershey.h>

/******************************************************************************/

static boolean	allocated	= FALSE;
static boolean	fontdefined	= FALSE;
static int	charheight	= DEFHEIGHT;
static int	charwidth	= DEFWIDTH;
static CHAR	curfont		= NULL;
static float	heightscale	= 1.0;
static float	widthscale	= 1.0;
static char	errmsg[512];

/******************************************************************************/

static int HersheyWarning( char* message )
{
   return( fprintf( stderr, "%s: %s\n", PROGNAME, message ) );
}

/******************************************************************************/

static void HersheyError( char* message, int status )
{
   fprintf( stderr, "%s: %s\n", PROGNAME, message );
   if ( allocated == TRUE )
      HersheyFreeFont( curfont );
   exit( status );
}

/******************************************************************************/

int HersheyCharWidth( int newval )
{

   if ( newval < 0 ) {
      sprintf( errmsg, "Out of range width value <%d> ignored", newval );
      HersheyWarning( errmsg );
      return( FAILURE );
   }

   charwidth = newval;
   widthscale = charwidth / (float)DEFWIDTH;
   return( newval );
}

/******************************************************************************/

int HersheyCharHeight( int newval )
{
   if ( newval < 0 ) {
      sprintf( errmsg, "Out of range height value <%d> ignored", newval );
      HersheyWarning( errmsg );
      return( FAILURE );
   }

   charheight = newval;
   heightscale = charheight / (float)DEFHEIGHT;
   return( newval );
}

/******************************************************************************/

int HersheyStrWidth( char* string )
{
   char *cptr;
   int width;

   for ( cptr = string, width = 0; *cptr; cptr++, width += charwidth );

   return( width );
}

/******************************************************************************/

int HersheyFreeFont( CHAR top )
{
   CHAR next, current;

   if ( top == NULL )
      return 0;

   current = top;

   while ( current != NULL ) {
      next = current->next;
      cfree( current );
      current = next;
   }

   return 1;
}

/******************************************************************************/

int HersheyFont( CHAR font )
{
   if ( font == NULL )
      return 0;

   fontdefined = TRUE;
   curfont = font;

   return 1;
}

/******************************************************************************/

CHAR HersheyLoadFont( char* fname )
{
   boolean chardone;
   CHAR current = NULL, last = NULL, retval = NULL;
   char action, inchar, line[512], *calloc(), *fgets();
   ENTRY eptr1, eptr2;
   ENTRYBLOCK strokes[MAX_STROKES_PER_CHAR];
   FILE *infile, *fopen();
   int x, y, xmin, xmax, ymin, ymax, npoints, loop;

   if ( NULL == ( infile = fopen( fname, "r" ) ) ) {
      sprintf( errmsg, "Unable to open file <%s> for reading", fname );
      HersheyWarning( errmsg );
      return( NULL );
   }

   while ( NULL != fgets( line, 512, infile ) ) {
      sscanf( line, "%c", &inchar );
      fgets( line, 512, infile );
      action = MOVE;  npoints = 0;  chardone = FALSE;  eptr1 = strokes;

      fgets( line, 512, infile );
      sscanf( line, "%d %d", &x, &y );
      xmin = xmax = x;  ymin = ymax = y;

      while ( npoints < MAX_STROKES_PER_CHAR && chardone == FALSE )
	    if ( x == -999 )
		switch( y ) {
		    case 0:
			action = MOVE;
			fgets( line, 512, infile );
			sscanf( line, "%d %d", &x, &y );
			break;
		    case -999:
			chardone = TRUE;
			break;
		    default:
			sprintf( errmsg,
			    "Found invalid (x,y) = (%d,%d) at character %c",
			    x, y, inchar );
			HersheyWarning( errmsg );
			return( retval );
			break;
		}
	    else {
		if ( x > xmax ) xmax = x;
		if ( x < xmin ) xmin = x;
		if ( y > ymax ) ymax = y;
		if ( y < ymin ) ymin = y;

		eptr1->action = action;
		eptr1->x = x;
		eptr1->y = y;

		if ( action == MOVE ) action = DRAW;
		npoints++;  eptr1++;
		fgets( line, 512, infile );
		sscanf( line, "%d %d", &x, &y );
	    }

	if ( ( current = (CHAR)calloc( 1, sizeof( CHARBLOCK ) 
	       + ( npoints - 1 ) * sizeof( ENTRYBLOCK ) ) )
	    == NULL ) {
	    sprintf( errmsg,
	    "Memory allocation failure at character <%c>", inchar );
	    HersheyWarning( errmsg );
	    return( retval );
      }

      if ( retval == NULL ) retval = last = current;

      current->ascii = inchar;
      current->length = npoints;
      current->xsize = xmax - xmin;
      current->ysize = ymax - ymin;
      current->next = NULL;

      for ( loop = 0, eptr1 = strokes, eptr2 = current->points;
      loop < current->length; loop++, eptr1++, eptr2++ ) {
         eptr2->action = eptr1->action;
         eptr2->x = eptr1->x;
         eptr2->y = eptr1->y;
      }

      if ( current != retval ) {
         last->next = current;
         last = current;
      }
   }
	
   allocated = TRUE;
   fclose( infile );
   return( retval );
}

/******************************************************************************/

int HersheyText( char* string, Coord curx, Coord cury, Coord curz )
{
   CHAR		current;
   char*	cptr;
   ENTRY	eptr;
   int		i, loop;

   if ( fontdefined == FALSE ) {
      HersheyWarning( "Must load font first" );
      return( FAILURE );
   }

   move( (Coord)curx, (Coord)cury, (Coord)curz );

   for ( loop = 0, cptr = string; *cptr; loop++, cptr++ ) {
      if ( *cptr == ' ' )
         curx += (float)DEFWIDTH * widthscale;
      else {
         for ( current = curfont; current != NULL && current->ascii != *cptr;
         current = current->next );

         if ( current == NULL ) return( loop );

         for ( i=0, eptr=current->points; i < current->length; i++, eptr++ ) {
            if ( eptr->action == DRAW )
               draw( (Coord)(curx + eptr->x * widthscale),
                     (Coord)(cury + eptr->y * heightscale),
                     (Coord)(curz) );
            else
               move( (Coord)(curx + eptr->x * widthscale),
                     (Coord)(cury + eptr->y * heightscale),
                     (Coord)(curz) );
         }
         curx += (float)DEFWIDTH * widthscale;
      }
   }

   return( loop );
}

/******************************************************************************/

int HersheyText2( char* string, Coord curx, Coord cury )
{
   CHAR		current;
   char*	cptr;
   ENTRY	eptr;
   int		i, loop;

   if ( fontdefined == FALSE ) {
      HersheyWarning( "Must load font first" );
      return( FAILURE );
   }

   move2( (Coord)curx, (Coord)cury );

   for ( loop = 0, cptr = string; *cptr; loop++, cptr++ ) {
      if ( *cptr == ' ' )
         curx += (float)DEFWIDTH * widthscale;
      else {
         for ( current = curfont; current != NULL && current->ascii != *cptr;
         current = current->next );

         if ( current == NULL ) return( loop );

#        ifdef VERSION_PS
            open_poly();
#        endif
         for ( i=0, eptr=current->points; i < current->length; i++, eptr++ ) {
            if ( eptr->action == DRAW )
               draw2( (Coord)(curx + eptr->x * widthscale),
                      (Coord)(cury + eptr->y * heightscale) );
            else {
#              ifdef VERSION_PS
                  close_poly();
                  open_poly();
#              endif
               move2( (Coord)(curx + eptr->x * widthscale),
                      (Coord)(cury + eptr->y * heightscale) );
            }
         }
#        ifdef VERSION_PS
            close_poly();
#        endif
         curx += (float)DEFWIDTH * widthscale;
      }
   }

   return( loop );
}

/******************************************************************************/
